<?php
/*
Plugin Name: Music Player for WooCommerce
Plugin URI: http://wordpress.dwbooster.com/content-tools/music-player-for-woocommerce
Version: 1.0.8
Text Domain: woocommerce_music_player
Author: <a href="http://www.codepeople.net">CodePeople</a>
Description: Music Player for WooCommerce includes the MediaElement.js music player in the pages of the products with audio files associated, and in the store's pages, furthermore, the plugin allows selecting between multiple skins.
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
 */

// CONSTANTS

define( 'WCMP_PLUGIN_URL', plugins_url( '', __FILE__ ) );
define( 'WCMP_DEFAULT_PLAYER_LAYOUT', 'mejs-classic' );
define( 'WCMP_DEFAULT_PLAYER_CONTROLS', 'default' );
define( 'WCMP_DEFAULT_PlAYER_TITLE', 1 );

// Load widgets
require_once 'widgets/playlist_widget.php';	

if ( !class_exists( 'WooCommerceMusicPlayer' ) ) {
	class WooCommerceMusicPlayer
	{
		//******************** ATTRIBUTES ************************
		
		private $_products_attrs = array();
		private $_player_layouts = array( 'mejs-classic', 'mejs-ted', 'mejs-wmp' );
		private $_player_controls= array( 'button', 'all', 'default' );
		private $_files_directory_path;
		private $_files_directory_url;
		private $_enqueued_resources = false;
		/**
		* WCMP constructor
		*
		* @access public
		* @return void	
		*/
		public function __construct()
		{
			// Generate upload dir
			$_files_directory = wp_upload_dir();
			$this->_files_directory_path = rtrim( $_files_directory[ 'basedir' ], '/' ).'/wcmp/';
			$this->_files_directory_url  = rtrim( $_files_directory[ 'baseurl' ], '/' ).'/wcmp/';
			
			if( !file_exists( $this->_files_directory_path ) ) @mkdir( $this->_files_directory_path, 0755 );
			add_action( 'plugins_loaded', array(&$this, 'load_textdomain') );
			add_action('init', array(&$this, 'init'), 99);
			add_action('admin_init', array(&$this, 'admin_init'), 99);
		} // End __constructor
		
		public function load_textdomain() 
		{
			load_plugin_textdomain( 'woocommerce_music_player', FALSE, basename( dirname( __FILE__ ) ) . '/languages/' );
		}
		
		public function get_product_attr( $product_id, $attr, $default = false )
		{
			if( !isset( $this->_products_attrs[ $product_id ] ) ) $this->_products_attrs[ $product_id ] = array();
			if( !isset( $this->_products_attrs[ $product_id ][ $attr ] ) )
			{	
				if( metadata_exists( 'post', $product_id, $attr ) )	
				{	
					$this->_products_attrs[ $product_id ][ $attr ] = get_post_meta( $product_id, $attr, true );
				}
				else
				{
					$this->_products_attrs[ $product_id ][ $attr ] = $default;
				}		
			}
			return $this->_products_attrs[ $product_id ][ $attr ];

		} // End get_product_attr
		
		//******************** WORDPRESS ACTIONS **************************
		
		public function init()
		{
			if( !is_admin() )
			{
				if( isset( $_REQUEST[ 'wcmp-action' ] ) && $_REQUEST[ 'wcmp-action' ] == 'play' )
				{
					if( isset( $_REQUEST[ 'wcmp-product' ] ) )
					{
						$product_id = @intval( $_REQUEST[ 'wcmp-product' ] );
						if( !empty( $product_id ) )
						{
							$product = wc_get_product( $product_id );
							if( $product !== false && isset( $_REQUEST[ 'wcmp-file' ] ) )
							{
								$files = $this->_get_product_files( 
									array( 
										'product' => $product, 
										'file_id' => $_REQUEST[ 'wcmp-file' ]
									) 
								);
								
								if( !empty( $files ) )
								{
									$file_url = $files[ $_REQUEST[ 'wcmp-file' ] ][ 'file' ];
									$this->_output_file( array( 'url' => $file_url ) );
								}	
							}	
						}	
					}
					exit;	
				}	
				else
				{
					add_action( 'woocommerce_shop_loop_item_title', array( &$this, 'include_main_player' ), 11 );
					add_action( 'woocommerce_before_add_to_cart_form', array( &$this, 'include_all_players' ), 11 );
				}	
			}	
		} // End init
		
		public function admin_init()
		{
			add_action( 'woocommerce_product_options_downloads',  array(&$this, 'woocommerce_product_options_downloads') );
			add_action('save_post', array(&$this, 'save_post'));
			add_action('delete_post', array(&$this, 'delete_post'));
			
		} // End admin_init
		
		public function save_post()
		{
			global $post;
			if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;
			if ( empty( $_POST['wcmp_nonce'] ) || !wp_verify_nonce( $_POST['wcmp_nonce'], 'session_id_'.session_id() ) ) return;
			if ( !isset($post) || 'product' !== $post->post_type || !current_user_can( 'edit_post', $post->ID ) ) return;
			
			$this->delete_post();
			
			// Save the player options
			$enable_player 	= ( isset( $_REQUEST[ '_wcmp_enable_player' ] ) ) ? 1 : 0;
			$player_style 	= ( 
					isset( $_REQUEST[ '_wcmp_player_layout' ] ) && 
					in_array( $_REQUEST[ '_wcmp_player_layout' ], $this->_player_layouts ) 
				) ? $_REQUEST[ '_wcmp_player_layout' ] : WCMP_DEFAULT_PLAYER_LAYOUT;
			
			$player_controls = ( 
					isset( $_REQUEST[ '_wcmp_player_controls' ] ) &&
					in_array( $_REQUEST[ '_wcmp_player_controls' ], $this->_player_controls )
				) ? $_REQUEST[ '_wcmp_player_controls' ] : WCMP_DEFAULT_PLAYER_CONTROLS;
			 
			$player_title = ( isset( $_REQUEST[ '_wcmp_player_title' ] ) ) ? 1 : 0;

			add_post_meta( $post->ID, '_wcmp_enable_player', $enable_player, true );
			add_post_meta( $post->ID, '_wcmp_player_layout', $player_style, true );
			add_post_meta( $post->ID, '_wcmp_player_controls', $player_controls, true );
			add_post_meta( $post->ID, '_wcmp_player_title', $player_title, true );
			
		} // End save_post
		
		public function delete_post()
		{
			global $post;
			if ( !isset($post) || 'product' !== $post->post_type || !current_user_can( 'edit_post', $post->ID ) ) return;
			
			delete_post_meta( $post->ID, '_wcmp_enable_player' );
			delete_post_meta( $post->ID, '_wcmp_player_layout' );
			delete_post_meta( $post->ID, '_wcmp_player_controls' );
			delete_post_meta( $post->ID, '_wcmp_player_title' );
			
			// Delete truncated version of the audio file
			$this->_delete_truncated_files( $post->ID );
			
		} // End delete_post
		
		public function enqueue_resources()
		{
			if( $this->_enqueued_resources ) return;
			$this->_enqueued_resources = true;	
			wp_enqueue_style( 'wp-mediaelement' );
			wp_enqueue_style( 'wp-mediaelement-skins', 'https://cdnjs.cloudflare.com/ajax/libs/mediaelement/2.21.2/mejs-skins.css' );
			wp_enqueue_style( 'wcmp-style', plugin_dir_url(__FILE__).'css/style.css' );
			wp_enqueue_script('jquery');
			wp_enqueue_script('wp-mediaelement');
			wp_enqueue_script('wcmp-script', plugin_dir_url(__FILE__).'js/public.js', array('jquery'));
		} // End enqueue_resources
		
		//******************** WOOCOMMERCE ACTIONS ************************
		
        /**
         * Load the additional attributes to select the player layout
         */
		public function woocommerce_product_options_downloads()
		{
			include_once 'views/player_options.php';	
		} // End woocommerce_product_options_downloads
		
		public function get_player(
			$audio_url, 
			$args = array()
		)
		{
			
			$default_args = array(
				'media_type' => 'mp3',
				'player_style' => WCMP_DEFAULT_PLAYER_LAYOUT,
				'player_controls' => WCMP_DEFAULT_PLAYER_CONTROLS
			);
			
			$args = array_merge( $default_args, $args );
			$id = ( !empty( $args[ 'id' ] ) ) ? 'id="'.esc_attr( $args[ 'id' ] ).'"' : '';
			
			return '<audio '.$id.' class="wcmp-player '.esc_attr($args[ 'player_controls' ]).' '.esc_attr($args[ 'player_style' ]).'"><source src="'.esc_url($audio_url).'" type="audio/'.esc_attr($args['media_type']).'" /></audio>';
			
		} // End get_player
		
		public function get_product_files( $id )
		{
			$product = wc_get_product( $id );
			if( !empty($product) )
				return $this->_get_product_files( array( 'product' => $product, 'all' => 1 ) );
			return array();
		}
		
		public function generate_audio_url( $product_id, $file_id )
		{
			return $this->_generate_audio_url( $product_id, $file_id );
		}
		
		public function include_main_player()
		{
			$product = wc_get_product();
			$files = $this->_get_product_files( array( 'product' => $product, 'first' => true ) );
			
			if( !empty( $files ) )
			{
				$this->enqueue_resources();
				
				$id = $product->get_id();
				$player_style 	= $this->get_product_attr( $id, '_wcmp_player_layout', WCMP_DEFAULT_PLAYER_LAYOUT );
				
				$player_controls = ( $this->get_product_attr( $id, '_wcmp_player_controls', WCMP_DEFAULT_PLAYER_CONTROLS ) != 'all' ) ? 'track' : '';
				
				foreach( $files as $index => $file )
				{
					$audio_url = $this->_generate_audio_url( $id,  $index );
					print '<div class="wcmp-player-container">'.$this->get_player($audio_url, array( 'player_controls' => $player_controls, 'player_style' => $player_style, 'media_type' => $file[ 'media_type' ] )).'</div>';
				}
			}	
		} // End include_main_player
		
		public function include_all_players()
		{
			$product = wc_get_product();
			$files = $this->_get_product_files( array( 'product' => $product, 'all' => true ) );
			$id = $product->get_id();
			
			if( !empty( $files ) )
			{
				$this->enqueue_resources();
				$player_style 		= $this->get_product_attr( $id, '_wcmp_player_layout', WCMP_DEFAULT_PLAYER_LAYOUT );
				$player_controls 	= $this->get_product_attr( $id, '_wcmp_player_controls', WCMP_DEFAULT_PLAYER_CONTROLS );
				$player_title 		= intval( $this->get_product_attr( $id, '_wcmp_player_title', WCMP_DEFAULT_PlAYER_TITLE ) );
				
				$counter = count( $files );
				if( $counter == 1 )
				{
					$player_controls = ($player_controls == 'button') ? 'track' : '';
					foreach( $files as $index => $file )
					{
						$audio_url = $this->_generate_audio_url( $id,  $index );
						print '<div class="wcmp-player-container">'.$this->get_player( $audio_url, array( 'player_controls' => $player_controls, 'player_style' => $player_style, 'media_type' => $file[ 'media_type' ] ) ).'</div>';
						
						if($player_title) 
							print '<div class="wcmp-player-title">'.$file['name'].'</div>';
					}	
				}	
				elseif( $counter > 1 )
				{
					$before = '<table class="wcmp-player-list">';
					$after  = '';
					foreach( $files as $index => $file )
					{
						$evenOdd = ( $counter % 2 == 1 ) ? 'wcmp-odd-row' : 'wcmp-even-row';
						$counter--;
						$audio_url = $this->_generate_audio_url( $id,  $index );
						print $before;
						$before = '';
						$after  = '</table>';
						if($player_controls != 'all' )
						{	
							print '<tr class="'.esc_attr( $evenOdd ).'"><td class="wcmp-player-container wcmp-column-player-'.esc_attr($player_style).'">'.$this->get_player( $audio_url, array( 'player_style' => $player_style, 'player_controls' => 'track', 'media_type' => $file[ 'media_type' ] ) ).'</td>';
							
							if($player_title) 
								print '<td class="wcmp-player-title wcmp-column-player-title">'.$file['name'].'</td>';
							
							print '</tr>';
						}
						else
						{
							print '<tr class="'.esc_attr( $evenOdd ).'"><td><span class="wcmp-player-container">'.$this->get_player( $audio_url, array( 'player_style' => $player_style, 'player_controls' => '', 'media_type' => $file[ 'media_type' ] ) ).'</span>';
							
							if($player_title)
								print '<span class="wcmp-player-title wcmp-column-player-title">'.$file['name'].'</span>';
							
							print '</td></tr>';
						}		
					}
					print $after;
				}
			}	
		} // End include_all_players
		
		//******************** PRIVATE METHODS ************************
		
		private function _generate_audio_url( $product_id, $file_index )
		{
			$url = $_SERVER['REQUEST_URI'];
			$url .= ( (strpos( $url, '?' ) === false) ? '?' : '&' ).'wcmp-action=play&wcmp-product='.$product_id.'&wcmp-file='.$file_index;
			return $url;
		} // End _generate_audio_url
		
		private function _delete_truncated_files( $product_id )
		{
			$files_arr = get_post_meta( $product_id, '_downloadable_files', true );
			if( !empty( $files_arr ) && is_array( $files_arr ) )
			{
				foreach( $files_arr as $file )
				{
					if( is_array( $file ) && !empty( $file[ 'file' ] ) )
					{
						$ext = pathinfo( $file[ 'file' ], PATHINFO_EXTENSION );
						$file_name = md5( $file[ 'file' ] ).( (!empty($ext) ) ? '.'.$ext : '' );
						@unlink( $this->_files_directory_path.$file_name );
					}	
				}
			}	
			
		} // End _delete_truncated_files
		
		/**
		 * Check if the file is an audio file and return its type or false
		 */
		private function _is_audio( $file_path )
		{
			if( preg_match( '/\.(mp3|ogg|oga|wav|wma|mp4|m4a)$/i', $file_path, $match ) )
				return $match[ 1 ];
			return false;
		} // End _is_audio
		
		private function _get_product_files( $args )
		{
			if( empty( $args[ 'product' ] ) ) return false;
			
			$product = $args[ 'product' ];
			$id = $product->get_id();
			if( !$this->get_product_attr( $id, '_wcmp_enable_player', false ) ) return false;
			
			$files = $product->get_files();
			if( empty( $files ) ) return false;
			
			$audio_files = array();
			foreach( $files as $index => $file )
			{
				if( !empty( $file[ 'file' ] ) && ($media_type = $this->_is_audio( $file[ 'file' ] )) !== false )
				{
					$file[ 'media_type' ] = $media_type;
					if( !empty( $args[ 'file_id' ] ) )
					{
						if( $args[ 'file_id' ] == $index )
						{
							$audio_files[ $index ] = $file;
							return $audio_files;
						}
					}
					elseif( !empty( $args[ 'first' ] ) )
					{
						$audio_files[ $index ] = $file;
						return $audio_files;
					}
					elseif( !empty( $args[ 'all' ] ) )
					{
						$audio_files[ $index ] = $file;
					}	
				}	
			}
			
			return $audio_files;
		} // End _get_product_files
		
		/**
		 * Create a temporal file and redirect to the new file
		 */
		private function _output_file( $args )
		{
			if( empty( $args[ 'url' ] ) ) return;
			$url = $args[ 'url' ];
			$file_extension = pathinfo($url, PATHINFO_EXTENSION);
			$file_name = md5( $url ).( ( !empty( $file_extension ) ) ? '.'.$file_extension : '' );
			$text = 'The requested URL was not found on this server';
			$file_path = $this->_files_directory_path.$file_name;
	
			if(file_exists($file_path)){
				header('location: '.$this->_files_directory_url.$file_name);
				exit;
			}else{
				try{	
					$c = false;
					if( ( $path = $this->_is_local( $url ) ) !== false ){
						$c = copy( $path, $file_path);
					}else{
						$response = wp_remote_get($resource, array( 'timeout' => WCMP_REMOTE_TIMEOUT, 'stream' => true, 'filename' => $url ) );
						if(  !is_wp_error( $response ) && $response['response']['code'] == 200 ) $c = true;
					}
					
					if( $c === true ){
						header('location: '.$this->_files_directory_url.$file_name);
						exit;
					}
				}
				catch(Exception $err)
				{
				}
				$text = 'Is not possible create the truncated file\'s version because the memory is not sufficient, or the permissions in the "uploads/wcmp" directory are not sufficient to create the truncated version of file.';
			}
			$this->_print_page_not_found( $text );
		} // End _output_file
		
		/**
		 * Print not found page if file it is not accessible
		 */
		private function _print_page_not_found( $text = 'The requested URL was not found on this server' ){
			header("Status: 404 Not Found");
			echo '<!DOCTYPE HTML PUBLIC "-//IETF//DTD HTML 2.0//EN">
				  <HTML><HEAD>
				  <TITLE>404 Not Found</TITLE>
				  </HEAD><BODY>
				  <H1>Not Found</H1>
				  <P>'.$text.'</P>
				  </BODY></HTML>
				 ';
		} // End _print_page_not_found
		
		private function _is_local( $url )
		{
			$attachment_id = attachment_url_to_postid( $url );
			if( $attachment_id )
			{
				$file_path = get_attached_file( $attachment_id );
				if( file_exists( $file_path ) ) return $file_path;
			}	
			return false;
		}
	} // End Class WooCommerceMusicPlayer
	
	$GLOBALS[ 'WooCommerceMusicPlayer' ] = new WooCommerceMusicPlayer;
}	 
?>